"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.uiToBackendReportDefinition = exports.getBackendReportState = exports.getBackendReportSource = void 0;
var _backendModel = require("../../../model/backendModel");
var _constants = require("../constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const uiToBackendReportDefinition = reportDefinition => {
  const {
    report_params: {
      report_name: reportName,
      description,
      report_source: reportSource,
      core_params: {
        base_url: baseUrl,
        time_duration: timeDuration,
        report_format: reportFormat,
        header,
        footer,
        limit,
        origin,
        timeFrom: timeFrom,
        timeTo: timeTo
      }
    },
    trigger,
    delivery
  } = reportDefinition;
  let backendReportDefinition = {
    name: reportName,
    isEnabled: getBackendIsEnabled(trigger),
    source: {
      description: description,
      type: getBackendReportSource(reportSource),
      id: getBackendReportSourceId(baseUrl),
      origin: origin
    },
    format: {
      duration: timeDuration,
      fileFormat: getBackendReportFormat(reportFormat),
      ...(limit && {
        limit: limit
      }),
      ...(header && {
        header: header
      }),
      ...(footer && {
        footer: footer
      }),
      ...(timeFrom && {
        timeFrom: timeFrom
      }),
      ...(timeTo && {
        timeTo: timeTo
      })
    },
    trigger: getBackendTrigger(trigger),
    ...(getBackendDelivery(delivery) && {
      delivery: getBackendDelivery(delivery)
    })
  };
  return backendReportDefinition;
};
exports.uiToBackendReportDefinition = uiToBackendReportDefinition;
const getBackendIsEnabled = trigger => {
  let enabled = true;
  if (trigger.trigger_params) {
    enabled = trigger.trigger_params.enabled;
  }
  return enabled;
};
const getBackendDelivery = delivery => {
  const {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  } = delivery;
  let res = {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  };
  return res;
};
const getBackendTrigger = trigger => {
  const {
    trigger_params: scheduleParams
  } = trigger;
  const {
    schedule
  } = {
    ...scheduleParams
  };
  let res = {
    triggerType: scheduleParams ? getBackendTriggerType(scheduleParams) : _backendModel.BACKEND_TRIGGER_TYPE.onDemand,
    schedule: schedule
  };
  return res;
};
const getBackendTriggerType = scheduleParams => {
  const {
    schedule_type: scheduleType
  } = scheduleParams;
  let res;
  switch (scheduleType) {
    case _constants.SCHEDULE_TYPE.cron:
      res = _backendModel.BACKEND_TRIGGER_TYPE.cronSchedule;
      break;
    case _constants.SCHEDULE_TYPE.recurring:
      res = _backendModel.BACKEND_TRIGGER_TYPE.intervalSchedule;
      break;
  }
  return res;
};
const getBackendReportFormat = reportFormat => {
  return _backendModel.REPORT_FORMAT_DICT[reportFormat];
};
const getBackendReportState = reportState => {
  return _backendModel.REPORT_STATE_DICT[reportState];
};
exports.getBackendReportState = getBackendReportState;
const getBackendReportSource = reportSource => {
  return _backendModel.REPORT_SOURCE_DICT[reportSource];
};
//TODO: tmp solution, we are extracting the id from the baseUrl, e.g. /app/dashboards#/view/<id>
// since currently dashboard/visualization id are not required in the UI model, will add in the future
exports.getBackendReportSource = getBackendReportSource;
const getBackendReportSourceId = baseUrl => {
  const id = baseUrl.split('/').pop() || '';
  return id;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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